#include "calculating.h"
#include <qmessagebox.h>

volatile bool stopped = false;

extensions_calculating::extensions_calculating(
    CalculatingWindow &w, const char *input_file_name,
    const char *good_vertex_file_name, const char *egraph_file_name,
    const char *good_sets_file_name, const char *max_exts_file_name) {
  type = no_result;
  this->w = &w;
  this->input_file_name = input_file_name;
  this->good_vertex_file_name = good_vertex_file_name;
  this->egraph_file_name = egraph_file_name;
  this->good_sets_file_name = good_sets_file_name;
  this->max_exts_file_name = max_exts_file_name;
  connect(this, SIGNAL(finished()), this, SLOT(done()));
  start();
}

void extensions_calculating::run() {
  stopped = false;

  try {
    ::maximal_extensions(*w, input_file_name.toStdString().data(),
                         good_vertex_file_name.toStdString().data(),
                         egraph_file_name.toStdString().data(),
                         good_sets_file_name.toStdString().data(),
                         max_exts_file_name.toStdString().data());
  } catch (const std::string &message) {
    type = std_string;
    result = QString::fromStdString(message);
  } catch (QString message) {
    type = qt_string;
    result = message;
  } catch (double x) {
    type = c_double;
  }
}

void extensions_calculating::done() {
  remove("workspace/extensions/cliques.txt");
  w->done();
}

cliques_calculating::cliques_calculating(CalculatingWindow &w,
                                         const char *egraph_file_name,
                                         const char *cliques_file_name) {
  type = no_result;
  this->w = &w;
  this->egraph_file_name = egraph_file_name;
  this->cliques_file_name = cliques_file_name;
  connect(this, SIGNAL(finished()), this, SLOT(done()));
  start();
}

void cliques_calculating::run() {
  stopped = false;

  try {
    ::maximal_cliques(*w, egraph_file_name.toStdString().data(),
                      cliques_file_name.toStdString().data());
  } catch (const string &message) {
    type = std_string;
    result = QString::fromStdString(message);
  } catch (QString message) {
    type = qt_string;
    result = message;
  } catch (double x) {
    type = c_double;
  }
}

void cliques_calculating::done() { w->done(); }

isomorphism_calculating::isomorphism_calculating(
    CalculatingWindow &w, const char *graphs_file_name,
    const char *classes_file_name) {
  type = no_result;
  this->w = &w;
  this->graphs_file_name = graphs_file_name;
  this->classes_file_name = classes_file_name;
  connect(this, SIGNAL(finished()), this, SLOT(done()));
  start();
}

void isomorphism_calculating::run() {
  stopped = false;

  try {
    ::graph_isomorphisms(*w, graphs_file_name.toStdString().data(),
                         classes_file_name.toStdString().data());
  } catch (const string &message) {
    type = std_string;
    result = QString::fromStdString(message);
  } catch (QString message) {
    type = qt_string;
    result = message;
  } catch (double x) {
    type = c_double;
  }
}

void isomorphism_calculating::done() { w->done(); }

CalculatingWindow::CalculatingWindow(QString caption, QWidget *parent)
    : QDialog(parent) {
  setupUi(this);
  setWindowTitle(caption);
  connect(pushButton, SIGNAL(clicked()), this, SLOT(stop_computing()));
  connect(this, SIGNAL(append(QString)), textEdit, SLOT(append(QString)));
  connect(this, SIGNAL(doUndo()), textEdit, SLOT(undo()));
  setWindowFlags(windowFlags() & ~Qt::WindowContextHelpButtonHint);
  show();
}

void CalculatingWindow::print(QString text) { emit append(text); }

void CalculatingWindow::undo() { emit doUndo(); }

void CalculatingWindow::done() {
  pushButton->setText("&Close");
  pushButton->setFocus();
  disconnect(pushButton, SIGNAL(clicked()), this, SLOT(stop_computing()));
  connect(pushButton, SIGNAL(clicked()), this, SLOT(close()));
  remove("maxexts.tmp");

  if (stopped) {
    QMessageBox::information(nullptr, "Computation stopped",
                             "Computation is stopped by user.");
  } else
    switch (calc->type) {
    case std_string:
      QMessageBox::critical(nullptr, "Computation failed", calc->result);
      break;
    case c_double:
      QMessageBox::information(nullptr, "Computation completed succefully",
                               "There is no extensions for given data");
      break;
    default:
      QMessageBox::information(nullptr, "Computation completed succefully",
                               succesful_message);
    }
}

void CalculatingWindow::stop_computing() {
  if (!stopped)
    stopped = true;
  else
    close();
}

void CalculatingWindow::closeEvent(QCloseEvent *event) { stop_computing(); }

extensions_window::extensions_window(const char *input_file_name,
                                     const char *good_vertex_file_name,
                                     const char *egraph_file_name,
                                     const char *good_sets_file_name,
                                     const char *max_exts_file_name)
    : CalculatingWindow("Maximal Extensions Computing") {
  succesful_message = "The maximal extensions of the given star complement "
                      "and the eigenvalue are computed successfuly.";
  calc = new extensions_calculating(*this, input_file_name,
                                    good_vertex_file_name, egraph_file_name,
                                    good_sets_file_name, max_exts_file_name);
}

cliques_window::cliques_window(const char *egraph_file_name,
                               const char *cliques_file_name)
    : CalculatingWindow("Maximal Cliques Computing") {
  succesful_message = "The maximal cliques of the given graph "
                      "and the eigenvalue are computed successfuly.";
  calc = new cliques_calculating(*this, egraph_file_name, cliques_file_name);
}

isomorphism_window::isomorphism_window(const char *graphs_file_name,
                                       const char *classes_file_name)
    : CalculatingWindow("Isomorphism Classes Computing") {

  succesful_message = "The isomorphism classes of the given graphs "
                      "are computed successfuly.";
  calc =
      new isomorphism_calculating(*this, graphs_file_name, classes_file_name);
}
